/*
 * InputModule.h
 *
 * Created 8/1/2009 By Johnny Huynh
 *
 * Version 00.00.01 8/1/2009
 *
 * Copyright Information:
 * All content copyright  2009 Johnny Huynh. All rights reserved.
 */
 
 #ifndef INPUTMODULE_H
 #define INPUTMODULE_H
 
 namespace InputModule {}
 namespace InputConfig {}
 
 #include "pandaFramework.h"
 #include "pandaSystem.h"
 
 #include "genericAsyncTask.h"
 #include "asyncTaskManager.h"
 
 #include "global.h"
 #include "CameraAdjustment.h"
 #include "Vector.h"
 #include "Math3D.h"
 
 // Rotation speed measured in degrees
 #define ROTATION_SPEED 1.55f
 
 namespace InputModule
 {
     inline void implement_keyboard_controls( WindowFramework& window );
     inline AsyncTask::DoneStatus apply_input_controls( GenericAsyncTask* task_Ptr, void* data_Ptr );
     template <typename T> inline void apply_first_person_controls( const T& fps_ratio_offset );
     template <typename T> inline void apply_third_person_controls( const T& fps_ratio_offset );
     template <typename T> inline void apply_movement( Ralph<T>* char_Ptr, const VECTOR2_TYPE& displacement );
     template <typename T> inline void apply_rotation( NodePath& node_path, const T& rotation_speed );
     inline VECTOR2_TYPE get_unit_xy_movement_direction( const PRIMARY_TYPE& degrees );
     inline VECTOR2_TYPE get_unit_xy_movement_direction( const VECTOR2_TYPE& dir );
     inline VECTOR2_TYPE get_unit_xy_movement_direction_standard( const PRIMARY_TYPE& degrees );
     inline VECTOR2_TYPE get_unit_xy_movement_direction_standard( const VECTOR2_TYPE& dir );
     inline void set_walk_key( const Event * event_Ptr, void * data_Ptr );
     inline void reset_walk_key( const Event * event_Ptr, void * data_Ptr );
     //inline void exitProgram( const Event * event_Ptr, void * data_Ptr );
     inline void start_move_forward( const Event * event_Ptr, void * data_Ptr );
     inline void stop_move_forward( const Event * event_Ptr, void * data_Ptr );
     inline void start_move_backward( const Event * event_Ptr, void * data_Ptr );
     inline void stop_move_backward( const Event * event_Ptr, void * data_Ptr );
     inline void start_move_left( const Event * event_Ptr, void * data_Ptr );
     inline void stop_move_left( const Event * event_Ptr, void * data_Ptr );
     inline void start_move_right( const Event * event_Ptr, void * data_Ptr );
     inline void stop_move_right( const Event * event_Ptr, void * data_Ptr );
     inline void start_rotate_left( const Event * event_Ptr, void * data_Ptr );
     inline void stop_rotate_left( const Event * event_Ptr, void * data_Ptr );
     inline void start_rotate_right( const Event * event_Ptr, void * data_Ptr );
     inline void stop_rotate_right( const Event * event_Ptr, void * data_Ptr );
 }
 
 namespace InputConfig
 {
    bool _move_forward( false );
    bool _move_backward( false );
    bool _move_left( false );
    bool _move_right( false );
    bool _rotate_left( false );
    bool _rotate_right( false );
    
    bool _prioritize_move_forward( true ); // else prioritize move backward
    bool _prioritize_move_left( true ); // else prioritize move right
    bool _prioritize_rotate_left( true ); // else prioritize rotate right
    
    void (Ralph< PRIMARY_TYPE >::*_move_forward_function_Ptr)() = &Ralph< PRIMARY_TYPE >::run_forward; // move (e.g. run or walk) function pointer
    
    std::string _close_window_key( "escape" );
    std::string _move_forward_key( "w" );
    std::string _move_backward_key( "s" );
    std::string _move_left_key( "a" );
    std::string _move_right_key( "d" );
    std::string _rotate_left_key( "q" );
    std::string _rotate_right_key( "e" );
    
    std::string _walk_key( "l" );//"shift" );
    bool _walk_key_on( false );
    
    std::string _standard_attack_key( "j" );
    
    enum ControlMode { THIRD_PERSON=0x0, FIRST_PERSON=0x1 };
    ControlMode _control_mode( THIRD_PERSON );
    PRIMARY_TYPE _third_person_camera_angle( ZERO ); // measured in degrees
 }
 
 /**
  * implement_keyboard_controls() implements the keyboard controls for manipulating
  * the global camera.
  *
  * @param (WindowFramework&) window
  */
 inline void InputModule::implement_keyboard_controls( WindowFramework& window )
 {
    // Enable Keyboard Detection
    window.enable_keyboard();
    
    global::_framework.define_key( InputConfig::_close_window_key, "Close window", &PandaFramework::event_esc, static_cast<void*>(&global::_framework) );
    //global::_framework.define_key( "escape", "Close window", &exitProgram, (void*) NULL );
    global::_framework.define_key( InputConfig::_move_forward_key, "Move forward", &InputModule::start_move_forward, static_cast<void*>(&global::_world_Ptr->get_camera()) );
    global::_framework.define_key( InputConfig::_move_forward_key + "-up", "Stop moving forward", &InputModule::stop_move_forward, static_cast<void*>(&global::_world_Ptr->get_camera()) );
    global::_framework.define_key( InputConfig::_move_backward_key, "Move backward", &InputModule::start_move_backward, static_cast<void*>(&global::_world_Ptr->get_camera()) );
    global::_framework.define_key( InputConfig::_move_backward_key + "-up", "Stop moving backward", &InputModule::stop_move_backward, static_cast<void*>(&global::_world_Ptr->get_camera()) );
    global::_framework.define_key( InputConfig::_move_left_key, "Move left", &InputModule::start_move_left, static_cast<void*>(&global::_world_Ptr->get_camera()) );
    global::_framework.define_key( InputConfig::_move_left_key + "-up", "Stop moving left", &InputModule::stop_move_left, static_cast<void*>(&global::_world_Ptr->get_camera()) );
    global::_framework.define_key( InputConfig::_move_right_key, "Move right", &InputModule::start_move_right, static_cast<void*>(&global::_world_Ptr->get_camera()) );
    global::_framework.define_key( InputConfig::_move_right_key + "-up", "Stop moving right", &InputModule::stop_move_right, static_cast<void*>(&global::_world_Ptr->get_camera()) );
    global::_framework.define_key( InputConfig::_rotate_left_key, "Rotate left", &InputModule::start_rotate_left, static_cast<void*>(NULL) );
    global::_framework.define_key( InputConfig::_rotate_left_key + "-up", "Stop rotating left", &InputModule::stop_rotate_left, static_cast<void*>(NULL) );
    global::_framework.define_key( InputConfig::_rotate_right_key, "Rotate right", &InputModule::start_rotate_right, static_cast<void*>(NULL) );
    global::_framework.define_key( InputConfig::_rotate_right_key + "-up", "Stop rotating right", &InputModule::stop_rotate_right, static_cast<void*>(NULL) );
    global::_framework.define_key( InputConfig::_walk_key, "Start walking", &InputModule::set_walk_key, static_cast<void*>(NULL) );
    global::_framework.define_key( InputConfig::_walk_key + "-up", "Start running", &InputModule::reset_walk_key, static_cast<void*>(NULL) );
    global::_framework.define_key( InputConfig::_standard_attack_key, "Standard attack", &Ralph<PRIMARY_TYPE>::execute_standard_attack, static_cast<void*>(static_cast<Ralph<PRIMARY_TYPE>*>(global::_player_1_Ptr)) );
    
    //printf( "%1d\n", EventHandler::get_global_event_handler()->has_hook( "shift-a" ) );
    
    global::_task_mgr_Ptr->add( new GenericAsyncTask( "Applies Input Controls", InputModule::apply_input_controls, static_cast<void*>(NULL) ) );
 }
 
 /**
  * apply_input_controls() applies the input controls every frame.
  *
  * @param (GenericAsyncTask*) task_Ptr
  * @param (void*) data_Ptr
  * @return AsyncTask::DoneStatus
  */
 inline AsyncTask::DoneStatus InputModule::apply_input_controls( GenericAsyncTask* task_Ptr, void* data_Ptr )
 {
    // if the player is can move
    if ( global::_player_1_Ptr->is_mobile() )
    {
        //global::_clock_Ptr->tick();
        //printf( "%1f\n", global::_clock_Ptr->get_frame_time() );
        PRIMARY_TYPE fps( global::_clock_Ptr->get_average_frame_rate() );
        PRIMARY_TYPE fps_ratio_offset( fps == ZERO ? ONE : BASE_FPS / fps );
        
        if ( InputConfig::_control_mode == InputConfig::FIRST_PERSON )
            InputModule::apply_first_person_controls( fps_ratio_offset );
        else // InputConfig::_control_mode == InputConfig::THIRD_PERSON
            InputModule::apply_third_person_controls( fps_ratio_offset );
    }
    else // the player cannot move
    {
        if ( InputConfig::_control_mode == InputConfig::FIRST_PERSON )
        {
            // Apply rotation
            InputModule::apply_rotation( *global::_player_1_Ptr, ROTATION_SPEED );
            
            // Readjust camera
            CameraAdjustment::adjust_camera_to_first_person_character( global::_world_Ptr->get_camera(), *global::_player_1_Ptr, CAMERA_XY_DISTANCE );
        }
        else // InputConfig::_control_mode == InputConfig::THIRD_PERSON
        {
            // Apply rotation
            InputModule::apply_rotation( global::_world_Ptr->get_camera(), ROTATION_SPEED );
            
            // Move the camera to a location that is the specified distance away from the character 
            // without changing the heading angle of the camera, and such that the camera is directly 
            // facing the character
            CameraAdjustment::adjust_camera_to_third_person_character( global::_world_Ptr->get_camera(), *global::_player_1_Ptr, CAMERA_XY_DISTANCE );
        }
    }
    
    return AsyncTask::DS_cont;
 }
 
 /**
  * apply_first_person_controls() applies first-person-like controls to the inputs.
  * The controlled character will still be viewed from the third-person perspective. 
  * Do not confuse this with a first-person camera view.
  *
  * @param (const T&) fps_ratio_offset - affects the movement rate of the inputs
  */
 template <typename T>
 inline void InputModule::apply_first_person_controls( const T& fps_ratio_offset )
 {
    // displacement is already normalized (multiplied by the fps_ratio_offset)
    VECTOR2_TYPE displacement( get_unit_xy_movement_direction( global::_player_1_Ptr->get_h() ) * fps_ratio_offset );
    
    // Apply movement
    InputModule::apply_movement( static_cast<Ralph<PRIMARY_TYPE>*>(global::_player_1_Ptr), displacement );
    
    nassertv( global::_player_1_Ptr != NULL );
    
    // Apply rotation
    InputModule::apply_rotation( *global::_player_1_Ptr, ROTATION_SPEED );
    
    // Apply animation
    if ( displacement.get_x() != ZERO || displacement.get_y() != ZERO )
    {
        (global::_player_1_Ptr->*InputConfig::_move_forward_function_Ptr)();
    }
    else
    {
        global::_player_1_Ptr->assume_neutral_position();
    }
    
    // Readjust camera
    CameraAdjustment::adjust_camera_to_first_person_character( global::_world_Ptr->get_camera(), *global::_player_1_Ptr, CAMERA_XY_DISTANCE );
 }
 
 /**
  * apply_third_person_controls() applies the standard third-person controls to the inputs.
  * It is coincident that the controlled character is viewed from the third-person
  * perspective. This has nothing to do with the third-person camera view; this strictly
  * applies to the player's controls.
  *
  * @param (const T&) fps_ratio_offset - affects the movement rate of the inputs
  */
 template <typename T>
 inline void InputModule::apply_third_person_controls( const T& fps_ratio_offset )
 {
    // displacement is a unit vector (multiplied by fps_ratio_offset)
    VECTOR2_TYPE displacement( InputModule::get_unit_xy_movement_direction( InputConfig::_third_person_camera_angle ) * fps_ratio_offset );
    
    // Apply movement
    InputModule::apply_movement( static_cast<Ralph<PRIMARY_TYPE>*>(global::_player_1_Ptr), displacement );
    
    NodePath& cam( global::_world_Ptr->get_camera() );
    
    // Apply rotation
    InputModule::apply_rotation( cam, ROTATION_SPEED );
    
    // Readjust camera and apply animation
    if ( displacement.get_x() != ZERO || displacement.get_y() != ZERO )
    {
        // Slowly readjust/spin the camera to be in back of the controlled character
        CameraAdjustment::adjust_camera_heading_gradually( cam, global::_player_1_Ptr->get_h(), 
             THIRD_PERSON_VIEW_READJUSTMENT_RATIO * fps_ratio_offset, THIRD_PERSON_VIEW_MIN_READJUSTMENT * fps_ratio_offset );
        
        global::_player_1_Ptr->look_at_xy_direction( displacement.get_x(), displacement.get_y() );
        
        (global::_player_1_Ptr->*InputConfig::_move_forward_function_Ptr)();
    }
    else
    {
        global::_player_1_Ptr->assume_neutral_position();
    }
    
    // Move the camera to a location that is the specified distance away from the character 
    // without changing the heading angle of the camera, and such that the camera is directly 
    // facing the character
    CameraAdjustment::adjust_camera_to_third_person_character( cam, *global::_player_1_Ptr, CAMERA_XY_DISTANCE );
 }
 
 /**
  * apply_movement() applies movement, based on the user's inputs and the specified 
  * movement direction offset by the frame-per-second ratio offset (fps_ratio_offset), 
  * to the specified Ralph.
  *
  * @param (Ralph<T>*) char_Ptr
  * @param (const VECTOR2_TYPE&) displacement
  */
 template <typename T>
 inline void InputModule::apply_movement( Ralph<T>* char_Ptr, const VECTOR2_TYPE& displacement )
 {
    // if displacement has occurred, update position
    if ( displacement.get_x() != ZERO )
    {
        char_Ptr->set_x( char_Ptr->get_x() + (displacement.get_x() * char_Ptr->get_movement_speed()) );
    }
    if ( displacement.get_y() != ZERO )
    {
        char_Ptr->set_y( char_Ptr->get_y() + (displacement.get_y() * char_Ptr->get_movement_speed()) );
    }
 }
 
 /**
  * apply_rotation() applies rotation, based on the user's inputs and the specified 
  * rotation_speed, to the specified NodePath.
  *
  * @param (NodePath&) node_path
  * @param (const T&) rotation_speed - measured in degrees (this should always be non-negative)
  */
 template <typename T>
 inline void InputModule::apply_rotation( NodePath& node_path, const T& rotation_speed )
 {
    if ( InputConfig::_rotate_left )
    {
        if ( !InputConfig::_rotate_right || InputConfig::_prioritize_rotate_left )
        {
            node_path.set_h( node_path.get_h() + rotation_speed );
        }
        else // InputConfig::_rotate_right && !InputConfig::_prioritize_rotate_left
        {
            node_path.set_h( node_path.get_h() - rotation_speed );
        }
    }
    else if ( InputConfig::_rotate_right && !InputConfig::_rotate_left)
    {
        node_path.set_h( node_path.get_h() - rotation_speed );
    }
 }
 
 /**
  * get_unit_xy_movement_direction() returns the unit vector representing the direction
  * of the user's input movement on the xy-plane based on the specified angle the 
  * actor is facing. The geographical coordinate system is assumed.
  *
  * @param (const PRIMARY_TYPE&) degrees
  * @return VECTOR2_TYPE
  */
 inline VECTOR2_TYPE InputModule::get_unit_xy_movement_direction( const PRIMARY_TYPE& degrees )
 {
    return InputModule::get_unit_xy_movement_direction( Vector::get_xy_direction( degrees ) );
 }
 
 /**
  * get_unit_xy_movement_direction() returns the unit vector representing the direction
  * of the user's input movement on the xy-plane based on the specified direction the 
  * actor is facing. The geographical coordinate system is assumed.
  *
  * @param (const VECTOR2_TYPE&) dir
  * @return VECTOR2_TYPE
  */
 inline VECTOR2_TYPE InputModule::get_unit_xy_movement_direction( const VECTOR2_TYPE& dir )
 {
    PRIMARY_TYPE displacement_x( ZERO );
    PRIMARY_TYPE displacement_y( ZERO );
    
    if ( InputConfig::_move_forward  )
    {
        if ( !InputConfig::_move_backward || InputConfig::_prioritize_move_forward )
        {
            displacement_x += dir.get_x();
            displacement_y += dir.get_y();
        }
        else // InputConfig::_move_backward && !InputConfig::_prioritize_move_forward
        {
            displacement_x -= dir.get_x();
            displacement_y -= dir.get_y();
        }
    }
    else if ( InputConfig::_move_backward && !InputConfig::_move_forward )
    {
        displacement_x -= dir.get_x();
        displacement_y -= dir.get_y();
    }
    
    // Geographical coordinate system
    // For left and right movement
    // Rotate the direction 90 degrees to get the left vector
    // crossProduct( (0, 0, 1), (-dir_x, dir_y, 0) )
    // = (-dir_y, dir_x, 0 )
    // PRIMARY_TYPE temp( dir_x );
    // dir_x = -dir_y;
    // dir_y = temp;
    
    // Standard xyz-coordinate system
    // For left and right movement
    // Rotate the direction 90 degrees to get the left vector
    // crossProduct( (0, 1, 0), (dir_x, 0, dir_z) )
    // = (dir_z, 0, -dir_x)
    // PRIMARY_TYPE temp( -dir_x );
    // dir_x = dir_z;
    // dir_z = temp;
    
    if ( InputConfig::_move_left )
    {
        if ( !InputConfig::_move_right || InputConfig::_prioritize_move_left )
        {
            //displacement_x += dir_x;
            //displacement_y += dir_y;
            displacement_x -= dir.get_y();
            displacement_y += dir.get_x();
        }
        else // InputConfig::_move_right && !InputConfig::_prioritize_move_left
        {
            //displacement_x -= dir_x;
            //displacement_y -= dir_y;
            displacement_x += dir.get_y();
            displacement_y -= dir.get_x();
        }
    }
    else if ( InputConfig::_move_right && !InputConfig::_move_left )
    {
        //displacement_x -= dir_x;
        //displacement_y -= dir_y;
        displacement_x += dir.get_y();
        displacement_y -= dir.get_x();
    }
    
    // normalize the displacements
    if ( displacement_x != ZERO || displacement_y != ZERO )
        Vector::normalize( displacement_x, displacement_y );
    
    return VECTOR2_TYPE( displacement_x, displacement_y );
 }
 
 /**
  * get_unit_xy_movement_direction_standard() returns the unit vector representing the direction
  * of the user's input movement on the xz-plane based on the specified angle the actor is facing. 
  * (albeit, we generalized the function to also work for a 2-dimensional world, hence, xy is in 
  * the function's name). The standard xyz-coordinate system is assumed.
  *
  * @param (const PRIMARY_TYPE&) degrees
  * @return VECTOR2_TYPE
  */
 inline VECTOR2_TYPE InputModule::get_unit_xy_movement_direction_standard( const PRIMARY_TYPE& degrees )
 {
    return InputModule::get_unit_xy_movement_direction_standard( Vector::get_xy_direction_standard( degrees ) );
 }
 
 /**
  * get_unit_xy_movement_direction_standard() returns the unit vector representing the direction
  * of the user's input movement on the xz-plane based on the specified direction the actor is facing. 
  * (albeit, we generalized the function to also work for a 2-dimensional world, hence, xy is in 
  * the function's name). The standard xyz-coordinate system is assumed.
  *
  * @param (const VECTOR2_TYPE&) dir
  * @return VECTOR2_TYPE
  */
 inline VECTOR2_TYPE InputModule::get_unit_xy_movement_direction_standard( const VECTOR2_TYPE& dir )
 {
    PRIMARY_TYPE displacement_x( ZERO );
    PRIMARY_TYPE displacement_y( ZERO );
    
    if ( InputConfig::_move_forward  )
    {
        if ( !InputConfig::_move_backward || InputConfig::_prioritize_move_forward )
        {
            displacement_x += dir.get_x();
            displacement_y += dir.get_y();
        }
        else // InputConfig::_move_backward && !InputConfig::_prioritize_move_forward
        {
            displacement_x -= dir.get_x();
            displacement_y -= dir.get_y();
        }
    }
    else if ( InputConfig::_move_backward && !InputConfig::_move_forward )
    {
        displacement_x -= dir.get_x();
        displacement_y -= dir.get_y();
    }
    
    // Geographical coordinate system
    // For left and right movement
    // Rotate the direction 90 degrees to get the left vector
    // crossProduct( (0, 0, 1), (-dir_x, dir_y, 0) )
    // = (-dir_y, dir_x, 0 )
    // PRIMARY_TYPE temp( dir_x );
    // dir_x = -dir_y;
    // dir_y = temp;
    
    // Standard xyz-coordinate system
    // For left and right movement
    // Rotate the direction 90 degrees to get the left vector
    // crossProduct( (0, 1, 0), (dir_x, 0, dir_z) )
    // = (dir_z, 0, -dir_x)
    // PRIMARY_TYPE temp( -dir_x );
    // dir_x = dir_z;
    // dir_z = temp;
    
    if ( InputConfig::_move_left )
    {
        if ( !InputConfig::_move_right || InputConfig::_prioritize_move_left )
        {
            //displacement_x += dir_x;
            //displacement_y += dir_y;
            displacement_x += dir.get_y();
            displacement_y -= dir.get_x();
        }
        else // InputConfig::_move_right && !InputConfig::_prioritize_move_left
        {
            //displacement_x -= dir_x;
            //displacement_y -= dir_y;
            displacement_x -= dir.get_y();
            displacement_y += dir.get_x();
        }
    }
    else if ( InputConfig::_move_right && !InputConfig::_move_left )
    {
        //displacement_x -= dir_x;
        //displacement_y -= dir_y;
        displacement_x -= dir.get_y();
        displacement_y += dir.get_x();
    }
    
    // normalize the displacements
    if ( displacement_x != ZERO || displacement_y != ZERO )
        Vector::normalize( displacement_x, displacement_y );
    
    return VECTOR2_TYPE( displacement_x, displacement_y );
 }
 
 /**
  * set_walk_key() turns the walk key on and off depending on if the
  * walk key was previously off or on respectively.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 /*inline void InputModule::set_walk_key( const Event * event_Ptr, void * data_Ptr )
 {
    InputConfig::_walk_key_on = !InputConfig::_walk_key_on;
    
    // if the walk_key has just been switched from off to on
    if ( InputConfig::_walk_key_on )
    {
        InputConfig::_move_forward_function_Ptr = &Ralph< PRIMARY_TYPE >::walk_forward;
    }
    else // the walk_key has just been switched from on to off
    {
        InputConfig::_move_forward_function_Ptr = &Ralph< PRIMARY_TYPE >::run_forward;
    }
 }*/
 
 /**
  * set_walk_key() turns the walk key on.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::set_walk_key( const Event * event_Ptr, void * data_Ptr )
 {
    InputConfig::_walk_key_on = true;
    
    InputConfig::_move_forward_function_Ptr = &Ralph< PRIMARY_TYPE >::walk_forward;
 }
 
 /**
  * reset_walk_key() turns the walk key off.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::reset_walk_key( const Event * event_Ptr, void * data_Ptr )
 {
    InputConfig::_walk_key_on = false;
    
    InputConfig::_move_forward_function_Ptr = &Ralph< PRIMARY_TYPE >::run_forward;
 }
 
 /**
  * exitProgram() exits the program.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 //inline void exitProgram( const Event * event_Ptr, void * data_Ptr )
 //{
    //exit( 0 );
 //}
 
 /**
  * start_move_forward() turns the _move_forward flag on.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::start_move_forward( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    NodePath* node_path_Ptr( static_cast<NodePath*>(data_Ptr) );
    
    InputConfig::_move_forward = true;
    
    if ( !InputConfig::_prioritize_move_forward )
        InputConfig::_prioritize_move_forward = true;
    
    if ( InputConfig::_control_mode == InputConfig::THIRD_PERSON )
        InputConfig::_third_person_camera_angle = node_path_Ptr->get_h();
 }
 
 /**
  * stop_move_forward() turns the _move_forward flag off.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::stop_move_forward( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    NodePath* node_path_Ptr( static_cast<NodePath*>(data_Ptr) );
    
    InputConfig::_move_forward = false;
    
    if ( InputConfig::_control_mode == InputConfig::THIRD_PERSON )
        InputConfig::_third_person_camera_angle = node_path_Ptr->get_h();
 }
 
 /**
  * start_move_backward() turns the _move_backward flag on.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::start_move_backward( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    NodePath* node_path_Ptr( static_cast<NodePath*>(data_Ptr) );
    
    InputConfig::_move_backward = true;
    
    if ( InputConfig::_prioritize_move_forward )
        InputConfig::_prioritize_move_forward = false;
    
    if ( InputConfig::_control_mode == InputConfig::THIRD_PERSON )
        InputConfig::_third_person_camera_angle = node_path_Ptr->get_h();
 }
 
 /**
  * stop_move_backward() turns the _move_backward flag off.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::stop_move_backward( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    NodePath* node_path_Ptr( static_cast<NodePath*>(data_Ptr) );
    
    InputConfig::_move_backward = false;
    
    if ( InputConfig::_control_mode == InputConfig::THIRD_PERSON )
        InputConfig::_third_person_camera_angle = node_path_Ptr->get_h();
 }
 
 /**
  * start_move_left() turns the _move_left flag on.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::start_move_left( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    NodePath* node_path_Ptr( static_cast<NodePath*>(data_Ptr) );
    
    InputConfig::_move_left = true;
    
    if ( !InputConfig::_prioritize_move_left )
        InputConfig::_prioritize_move_left = true;
    
    if ( InputConfig::_control_mode == InputConfig::THIRD_PERSON )
        InputConfig::_third_person_camera_angle = node_path_Ptr->get_h();
 }
 
 /**
  * stop_move_left() turns the _move_left flag off.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::stop_move_left( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    NodePath* node_path_Ptr( static_cast<NodePath*>(data_Ptr) );
    
    InputConfig::_move_left = false;
    
    if ( InputConfig::_control_mode == InputConfig::THIRD_PERSON )
        InputConfig::_third_person_camera_angle = node_path_Ptr->get_h();
 }
 
 /**
  * start_move_right() turns the _move_right flag on.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::start_move_right( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    NodePath* node_path_Ptr( static_cast<NodePath*>(data_Ptr) );
    
    InputConfig::_move_right = true;
    
    if ( InputConfig::_prioritize_move_left )
        InputConfig::_prioritize_move_left = false;
    
    if ( InputConfig::_control_mode == InputConfig::THIRD_PERSON )
        InputConfig::_third_person_camera_angle = node_path_Ptr->get_h();
 }
 
 /**
  * stop_move_right() turns the _move_right flag off.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::stop_move_right( const Event * event_Ptr, void * data_Ptr )
 {
    nassertv( data_Ptr != NULL );
    NodePath* node_path_Ptr( static_cast<NodePath*>(data_Ptr) );
    
    InputConfig::_move_right = false;
    
    if ( InputConfig::_control_mode == InputConfig::THIRD_PERSON )
        InputConfig::_third_person_camera_angle = node_path_Ptr->get_h();
 }
 
 /**
  * start_rotate_left() turns the _rotate_left flag on.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::start_rotate_left( const Event * event_Ptr, void * data_Ptr )
 {
    InputConfig::_rotate_left = true;
    
    if ( !InputConfig::_prioritize_rotate_left )
        InputConfig::_prioritize_rotate_left = true;
 }
 
 /**
  * stop_rotate_left() turns the _rotate_left flag off.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::stop_rotate_left( const Event * event_Ptr, void * data_Ptr )
 {
    InputConfig::_rotate_left = false;
 }
 
 /**
  * start_rotate_right() turns the _rotate_right flag on.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::start_rotate_right( const Event * event_Ptr, void * data_Ptr )
 {
    InputConfig::_rotate_right = true;
    
    if ( InputConfig::_prioritize_rotate_left )
        InputConfig::_prioritize_rotate_left = false;
 }
 
 /**
  * stop_rotate_right() turns the _rotate_right flag off.
  *
  * @param (const Event *) event_Ptr
  * @param (void *) data_Ptr
  */
 inline void InputModule::stop_rotate_right( const Event * event_Ptr, void * data_Ptr )
 {
    InputConfig::_rotate_right = false;
 }
 
 #undef ROTATION_SPEED
 
 #endif // INPUTMODULE_H